// ERP scraper: robust extraction using hidden input names, auto-scrape on change, and minimize UI
(function() {
  if (window.__erp_scraper_installed) return;
  window.__erp_scraper_installed = true;

  const $ = (s, r=document) => r.querySelector(s);
  const $$ = (s, r=document) => Array.from(r.querySelectorAll(s));

  // Create UI
  const panel = document.createElement('div');
  panel.id = 'erp-scraper-panel';
  panel.innerHTML = `
    <div class="erp-header">
      <span class="title">ERP Scraper</span>
      <button id="erp-minimize" title="Minimize">━</button>
    </div>
    <div class="erp-body">
      <div class="meta">Meta: <span class="meta-text">—</span></div>
      <div class="controls">
        <button id="erp-refresh" title="Refresh page data">⟳ Refresh</button>
        <button id="erp-scan" title="Scrape table">🔍 Scrape</button>
        <button id="erp-download" title="Download CSV">⬇️ Download</button>
      </div>
      <div class="status">Idle</div>
      <div class="counter">Scraped: <span class="count">0</span></div>
      <div class="preview-label">CSV Preview</div>
      <textarea class="csv-preview" readonly placeholder="CSV will appear here after scraping..."></textarea>
    </div>
  `;
  document.body.appendChild(panel);

  const statusEl = $('.status', panel);
  const countEl = $('.count', panel);
  const metaTextEl = $('.meta-text', panel);
  const previewTa = $('.csv-preview', panel);
  const minimizeBtn = $('#erp-minimize', panel);

  let lastScrape = { rows: 0, csv: '', meta: {} };

  function setStatus(t) { statusEl.textContent = t; }

  function findTable() {
    return $('table.table.table-bordered') || $('table');
  }

  function readMeta() {
    const box = $('.alert.alert-info');
    if (!box) return {};
    const ids = ['room_info','course_info','subject_name_info','subject_code_info','batch_info','semester_info','currentDate','currentTime'];
    const out = {};
    for (const id of ids) { const el = document.getElementById(id); if (el) out[id] = el.innerText.trim(); }
    return out;
  }

  function extractRowsUsingHiddenInputs(table) {
    // For each tr, collect hidden inputs with names like students[1][aadhar_no]
    const trs = $$('tbody tr', table);
    if (!trs.length) return [];
    const rows = [];
    for (const tr of trs) {
      const row = {};
      // find all hidden inputs inside this tr
      const inputs = $$('input[type="hidden"]', tr);
      for (const inp of inputs) {
        const name = inp.getAttribute('name');
        const m = name && name.match(/students\[(\d+)\]\[(.+)\]/);
        if (m) {
          const key = m[2];
          row[key] = inp.value.trim();
        }
      }
      // fallback: read visible tds if hidden inputs missing
      if (!Object.keys(row).length) {
        const tds = $$('td', tr).map(td => td.innerText.trim());
        // map by position if possible
        row['_tds'] = tds;
      }
      rows.push(row);
    }
    return rows;
  }

  function canonicalRowFromExtracted(obj, index) {
    // canonical order: Sr. No, Aadhaar(aadhar_no), Name(student_name), Father(father_name), Phone(phone_no), Address(address), Hostelite(hosteladd), Attendance(status)
    // Use sequential serial number (index+1) to avoid period/room values being used as Sr. No
    const sr = (index + 1).toString();
    const aadhar = obj['aadhar_no'] || obj['aadhar'] || obj['aadhar_no'] || '';
    const name = obj['student_name'] || obj['student'] || '';
    const father = obj['father_name'] || obj['father'] || '';
    const phone = obj['phone_no'] || obj['phone'] || '';
    const address = obj['address'] || '';
    const hostel = obj['hosteladd'] || obj['hostel'] || '';
    const attendance = obj['status'] || '';
    // if obj._tds present, try to pick positions
    if (obj['_tds']) {
      const t = obj['_tds'];
      return [ t[0]||'', t[1]||'', t[2]||'', t[3]||'', t[4]||'', t[5]||'', t[6]||'', t[7]||'' ];
    }
    return [ sr, aadhar, name, father, phone, address, hostel, attendance ];
  }

  function buildCSV(meta, rows) {
    const esc = v => '"' + String(v || '').replace(/"/g, '""') + '"';
    const lines = [];
    if (meta && meta.course_info) {
      lines.push([esc('Course'), esc(meta.course_info)].join(','));
      if (meta.subject_name_info) lines.push([esc('Subject'), esc(meta.subject_name_info + (meta.subject_code_info ? ' ('+meta.subject_code_info+')' : ''))].join(','));
      if (meta.batch_info) lines.push([esc('Batch'), esc(meta.batch_info)].join(','));
      if (meta.currentDate) lines.push([esc('Date'), esc((meta.currentDate||'') + ' | ' + (meta.currentTime||''))].join(','));
      lines.push('');
    }
    const header = ['Sr. No','Aadhaar','Name','Father','Phone','Address','Hostelite','Attendance'];
    lines.push(header.map(esc).join(','));
    rows.forEach((r,i) => {
      const cr = canonicalRowFromExtracted(r,i);
      lines.push(cr.map(esc).join(','));
    });
    return lines.join('\n');
  }

  async function doScrapeAndStore() {
    setStatus('Scanning...');
    const table = findTable();
    if (!table) { setStatus('No table found'); return; }
    const meta = readMeta();
    metaTextEl.textContent = meta.course_info ? `${meta.course_info} | ${meta.subject_name_info || ''} ${meta.subject_code_info ? '('+meta.subject_code_info+')':''}` : '—';
    const rows = extractRowsUsingHiddenInputs(table);
    const csv = buildCSV(meta, rows);
    lastScrape = { rows: rows.length, csv, meta };
    countEl.textContent = rows.length;
    previewTa.value = csv.slice(0,20000);
    setStatus('Scrape complete — ' + rows.length + ' rows');
    return lastScrape;
  }

  function downloadCSV(filename) {
    if (!lastScrape.csv) { setStatus('Nothing to download — scrape first'); return; }
    setStatus('Downloading...');
    const fname = filename || buildFilenameFromMeta(lastScrape.meta) || makeFilename();
    chrome.runtime.sendMessage({ type: 'download', data: lastScrape.csv, filename: fname }, ()=> setStatus('Download started'));
  }

  function makeFilename() { return `erp-scrape-${(new Date()).toISOString().slice(0,19).replace(/:/g,'-')}.csv`; }

  // Build a descriptive filename using meta info: course, batch, group, semester
  function sanitize(s) {
    if (!s) return '';
    return s.replace(/\s+/g,'_').replace(/[^0-9A-Za-z_\-]/g,'').replace(/_+/g,'_');
  }

  function buildFilenameFromMeta(meta) {
    // meta.course_info example: "B.Tech. Hons.(CSE) (Group C)"
    const course = meta.course_info || '';
    const batch = meta.batch_info || '';
    const semester = meta.semester_info || '';
    let group = '';
    const gmatch = course.match(/Group\s*([A-Za-z0-9]+)/i);
    if (gmatch) group = gmatch[1];
    // fallback: try to find '(Group X)' in course
    const parts = [course, batch, group ? ('Group'+group) : '', semester ? ('Sem'+semester) : ''].map(sanitize).filter(Boolean);
    const when = (new Date()).toISOString().slice(0,19).replace(/:/g,'-');
    return (parts.length ? parts.join('_') + '_' : '') + when + '.csv';
  }

  // Buttons
  $('#erp-scan', panel).addEventListener('click', async () => { $('#erp-scan', panel).classList.add('pulse'); await doScrapeAndStore(); setTimeout(()=>$('#erp-scan', panel).classList.remove('pulse'), 600); });
  $('#erp-download', panel).addEventListener('click', () => downloadCSV());
  $('#erp-refresh', panel).addEventListener('click', async () => { setStatus('Refreshing...'); panel.classList.add('rotating'); const sel = document.getElementById('subject_select'); if (sel) sel.dispatchEvent(new Event('change',{bubbles:true})); await new Promise(r=>setTimeout(r,900)); panel.classList.remove('rotating'); await doScrapeAndStore(); });

  // minimize/restore
  minimizeBtn.addEventListener('click', () => {
    panel.classList.toggle('minimized');
    if (panel.classList.contains('minimized')) {
      minimizeBtn.textContent = '+';
    } else { minimizeBtn.textContent = '━'; }
  });

  // draggable
  panel.addEventListener('mousedown', e => {
    if (e.target.tagName === 'BUTTON') return;
    const startX = e.clientX, startY = e.clientY;
    const rect = panel.getBoundingClientRect();
    function move(ev) { panel.style.right='auto'; panel.style.left = (rect.left + (ev.clientX - startX)) + 'px'; panel.style.top = (rect.top + (ev.clientY - startY)) + 'px'; }
    function up(){ document.removeEventListener('mousemove', move); document.removeEventListener('mouseup', up); }
    document.addEventListener('mousemove', move); document.addEventListener('mouseup', up);
  });

  // Auto-scrape: watch for subject select changes or table updates
  const debounced = (fn, wait=600) => { let t; return (...a) => { clearTimeout(t); t = setTimeout(()=>fn(...a), wait); }; };
  const debouncedScrape = debounced(() => doScrapeAndStore(), 600);

  // observe subject select
  const subj = document.getElementById('subject_select');
  if (subj) subj.addEventListener('change', debouncedScrape);

  // observe table container for mutations
  const table = findTable();
  if (table) {
    const mo = new MutationObserver(debouncedScrape);
    mo.observe(table, { childList: true, subtree: true, attributes: true });
  }

  // initial
  setTimeout(()=>doScrapeAndStore(), 700);

})();
// Content script: inject control panel and scraping logic
(function() {
  if (window.__erp_scraper_installed) return;
  window.__erp_scraper_installed = true;

  // Helpers
  function qs(sel, root=document) { return root.querySelector(sel); }
  function qsa(sel, root=document) { return Array.from(root.querySelectorAll(sel)); }

  // Create UI container
  const panel = document.createElement('div');
  panel.id = 'erp-scraper-panel';
  panel.innerHTML = `
    <div class="erp-header">ERP Scraper</div>
    <div class="erp-body">
      <div class="meta"></div>
      <div class="controls">
        <button id="erp-refresh">Refresh & Start</button>
        <button id="erp-scan">Scrape Now</button>
        <button id="erp-download">Download CSV</button>
      </div>
      <div class="status">Idle</div>
      <div class="counter">Scraped: <span class="count">0</span></div>
    </div>
  `;

  document.body.appendChild(panel);

  // Styles will be applied from styles.css

  const status = qs('.status', panel);
  const counter = qs('.count', panel);
  const meta = qs('.meta', panel);

  let lastScrape = { rows: 0, csv: '' };

  function setStatus(s) { status.textContent = s; }

  function findTable() {
    // Try to find the main attendance table by headings
    const tables = qsa('table');
    for (const t of tables) {
      const text = t.innerText.toLowerCase();
      if (text.includes('aadhaar') && text.includes('attendance')) return t;
      // fallback: look for 'sr. no' and 'name'
      if (text.includes('sr. no') && text.includes('name')) return t;
    }
    return tables[0] || null;
  }

  function scrape() {
    setStatus('Scanning page...');
    const table = findTable();
    if (!table) {
      setStatus('No table found on this page');
      return { meta: {}, rows: [] };
    }

    // Attempt to read meta info from page text near the select dropdown or info box
    const metaBox = qsa('div, p').find(el => el.innerText && el.innerText.match(/Course:|Subject:|Batch:|Period:|Semester:|Date:/i));
    const metaText = metaBox ? metaBox.innerText.trim() : '';

    // Rows
    const headers = qsa('th', table).map(h => h.innerText.trim())
      .filter(Boolean);
    const rows = qsa('tr', table).slice(1).map(tr => qsa('td', tr).map(td => td.innerText.trim()));

    // Normalize rows (ensure same length)
    const maxCols = Math.max(headers.length, ...rows.map(r => r.length));
    const normalizedRows = rows.map(r => {
      const copy = r.slice();
      while (copy.length < maxCols) copy.push('');
      return copy;
    });

    // Update UI
    counter.textContent = normalizedRows.length;
    meta.textContent = metaText;
    setStatus('Found table — ready');

    const headerRow = headers.length ? headers : Array.from({length: maxCols}, (_,i) => 'col' + (i+1));

    return { meta: parseMeta(metaText), headers: headerRow, rows: normalizedRows };
  }

  function parseMeta(text) {
    const out = {};
    if (!text) return out;
    const map = { 'course': /Course:\s*(.+)/i, 'subject': /Subject:\s*(.+)/i, 'batch': /Batch:\s*(.+)/i, 'period': /Period:\s*(.+)/i, 'semester': /Semester:\s*(.+)/i, 'date': /Date:\s*(.+)/i };
    for (const k in map) {
      const m = text.match(map[k]);
      if (m) out[k] = m[1].trim();
    }
    return out;
  }

  function toCSV(headers, rows, meta) {
    const esc = v => '"' + String(v).replace(/"/g, '""') + '"';
    const lines = [];
    // meta as top rows
    if (meta && Object.keys(meta).length) {
      for (const k of Object.keys(meta)) {
        lines.push([esc(k), esc(meta[k])].join(','));
      }
      lines.push('');
    }
    lines.push(headers.map(esc).join(','));
    for (const r of rows) lines.push(r.map(esc).join(','));
    return lines.join('\n');
  }

  async function doScrapeAndStore() {
    const { meta, headers, rows } = scrape();
    const csv = toCSV(headers, rows, meta);
    lastScrape = { rows: rows.length, csv, headers };
    setStatus('Scrape complete');
    return lastScrape;
  }

  function downloadCSV(filename) {
    if (!lastScrape.csv) {
      setStatus('Nothing to download — please scrape first');
      return;
    }
    setStatus('Preparing download...');
    chrome.runtime.sendMessage({ type: 'download', data: lastScrape.csv, filename: filename || 'erp-scrape.csv' }, () => {
      setStatus('Download started');
    });
  }

  // Buttons
  qs('#erp-scan', panel).addEventListener('click', async () => {
    qs('#erp-scan', panel).classList.add('pulse');
    await doScrapeAndStore();
    setTimeout(() => qs('#erp-scan', panel).classList.remove('pulse'), 600);
  });

  qs('#erp-download', panel).addEventListener('click', () => {
    downloadCSV(makeFilename());
  });

  qs('#erp-refresh', panel).addEventListener('click', async () => {
    setStatus('Refreshing page...');
    // simple visual animation
    panel.classList.add('rotating');
    // try to trigger any site filtering/selectors by dispatching an event: click the first select if present
    const sel = qs('select');
    if (sel) {
      sel.dispatchEvent(new Event('change', { bubbles: true }));
    }
    // wait a bit for page to update
    await new Promise(r => setTimeout(r, 1200));
    panel.classList.remove('rotating');
    await doScrapeAndStore();
  });

  function makeFilename() {
    const m = lastScrape && lastScrape.headers ? (new Date()).toISOString().slice(0,19).replace(/:/g,'-') : 'erp-scrape';
    return `erp-scrape-${m}.csv`;
  }

  // small draggable behaviour
  panel.addEventListener('mousedown', e => {
    if (e.target.id === 'erp-refresh' || e.target.tagName === 'BUTTON') return;
    const startX = e.clientX, startY = e.clientY;
    const rect = panel.getBoundingClientRect();
    function move(ev) {
      panel.style.right = 'auto';
      panel.style.left = (rect.left + (ev.clientX - startX)) + 'px';
      panel.style.top = (rect.top + (ev.clientY - startY)) + 'px';
    }
    function up() { document.removeEventListener('mousemove', move); document.removeEventListener('mouseup', up); }
    document.addEventListener('mousemove', move); document.addEventListener('mouseup', up);
  });

  // initial auto-scan
  setTimeout(() => { doScrapeAndStore(); }, 800);

})();
